// /** @type {import('next').NextConfig} */
//
// const nextConfig = {
//     // Enable styled-components support in the Next.js compiler
//     compiler: {
//         styledComponents: true,
//     },
//
//     // Set React strict mode (false in this case)
//     reactStrictMode: false,
//
//     // Configure allowed domains for optimized images
//     images: {
//         domains: ['bestinbd.com', 'localhost'],
//     },
//
//     // Custom Webpack configuration
//     webpack: (config, { dev }) => {
//         if (config.cache && !dev) {
//             config.cache = {
//                 type: 'memory',
//             };
//         }
//         // Important: return the modified config
//         return config;
//     },
//
//     // Ignore ESLint errors during builds
//     eslint: {
//         ignoreDuringBuilds: true,
//     },
//
//     // Enable SWC minification for faster builds
//     swcMinify: true,
//
//     // Optimize fonts loading
//     optimizeFonts: true,
//
//     // Set up custom headers for security
//     async headers() {
//         return [
//             {
//                 source: '/(.*)',
//                 headers: [
//                     {
//                         key: 'X-Frame-Options',
//                         value: 'DENY',
//                     },
//                     {
//                         key: 'X-Content-Type-Options',
//                         value: 'nosniff',
//                     },
//                     {
//                         key: 'Referrer-Policy',
//                         value: 'strict-origin-when-cross-origin',
//                     },
//                     {
//                         key: 'Content-Security-Policy',
//                         value: "default-src 'self'; img-src *; media-src *; script-src 'self' 'unsafe-inline' 'unsafe-eval'; style-src 'self' 'unsafe-inline'; frame-ancestors 'none';",
//                     },
//                 ],
//             },
//         ];
//     },
// };
//
// export default nextConfig;


/** @type {import('next').NextConfig} */

const nextConfig = {
    // Enable styled-components support in the Next.js compiler
    compiler: {
        styledComponents: true,
    },
    output: 'standalone',
    crossOrigin: 'anonymous',
    // Set React strict mode (false in this case)
    reactStrictMode: false,

    // Configure allowed domains for optimized images
    images: {
        domains: ['project.bestinbd.com', 'localhost'],
        minimumCacheTTL: 60, // Cache images for 60 seconds
    },
    transpilePackages: ['@studio-freight/compono'],

    // Custom Webpack configuration
    webpack: (config, { dev }) => {
        if (config.cache && !dev) {
            config.cache = {
                type: 'memory',
            };
        }
        return config;
    },

    // Ignore ESLint errors during builds
    eslint: {
        ignoreDuringBuilds: true,
    },

    // Optimize fonts loading
    optimizeFonts: true,
    experimental: {
        scrollRestoration: true, // Preserve scroll position on route changes
        reactCompiler: true,
    },
    // Caching for API Routes (Edge and serverless functions)
    // Configuring headers for better caching of API routes and assets
    async headers() {
        return [
            {
                source: '/api/:path*',
                headers: [
                    { key: 'Cache-Control', value: 's-maxage=86400, stale-while-revalidate=59' }, // Cache API responses for 1 day
                ],
            },
            {
                source: '/public/images/static/:path*',
                headers: [
                    { key: 'Cache-Control', value: 'public, max-age=31536000, immutable' }, // Cache static assets long-term
                ],
            },
        ];
    },


    // Enable SWC minification for faster builds
    swcMinify: true,

    // Bundling optimizations for faster route loading
    optimizeCss: true,   // Enable CSS optimization
    productionBrowserSourceMaps: true,

};

export default nextConfig;


