'use client';
import React, { useState, useEffect, useRef } from 'react';
import styled from "styled-components";
import { Container, Row, Col } from "react-bootstrap";
import Select from 'react-select';
import {
    LightgalleryProvider,
    LightgalleryItem,
} from "react-lightgallery";
import "lightgallery.js/dist/css/lightgallery.css";
import { gsap } from "gsap";
import reactHtmlParser from "react-html-parser";
import Image from 'next/image';

const Gallery = ({ data }) => {
    const filteredOptions = data?.filter(f => f?.page_data?.business_type && f?.page_data?.business_type.trim() !== '');
    const [selectedFilter, setSelectedFilter] = useState(
        filteredOptions.length > 0 ? filteredOptions[0] : ''
    );
    const [isMobile, setIsMobile] = useState(false);
    const imageRefs = useRef([]);

    // Create options for react-select
    const selectOptions = filteredOptions.map(option => ({
        value: option,
        label: reactHtmlParser(option?.page_data?.business_type)
    }));

    useEffect(() => {
        const checkMobile = () => {
            setIsMobile(window.innerWidth <= 767);
        };

        checkMobile();
        window.addEventListener('resize', checkMobile);
        return () => window.removeEventListener('resize', checkMobile);
    }, []);

    const PhotoItem = ({ image, thumb, group, index }) => (
        <LightgalleryItem group={group} src={image} thumb={thumb}>
            <div className="image-wrapper">
                <Image
                    src={image}
                    blurDataURL="data:image/jpeg;base64,/9j/4AAQSkZJRgABAQE..."
                    placeholder={"blur"}
                    objectFit={'cover'}
                    layout={'fill'}
                    ref={el => imageRefs.current[index] = el} />
            </div>
        </LightgalleryItem>
    );

    const filteredData = selectedFilter?.sections?.filter(f => f?.section_data?.template === 'gallery')?.[0]?.images?.list?.slice(0, 8);

    useEffect(() => {
        if (filteredData?.length > 0) {
            gsap.fromTo(imageRefs.current,
                { opacity: 0, y: 50 },
                { opacity: 1, y: 0, duration: 0.5, stagger: 0.2, ease: "power2.out" }
            );
        }
    }, [filteredData]);

    const handleSelectChange = (selectedOption) => {
        setSelectedFilter(selectedOption.value);
    };

    return (
        <StyledGallery className='pt-150 pb-150'>
            <Container className=''>
                {isMobile ? (
                    <StyledSelect
                        options={selectOptions}
                        value={selectOptions.find(option => option.value === selectedFilter)}
                        onChange={handleSelectChange}
                        classNamePrefix="react-select"
                        isSearchable={false}
                    />
                ) : (
                    <ul>
                        {filteredOptions && filteredOptions?.length > 0 && filteredOptions.map((option, index) => (
                            <li
                                key={index}
                                className={selectedFilter === option ? 'active' : ''}
                                onClick={() => setSelectedFilter(option)}
                            >
                                <p>{reactHtmlParser(option?.page_data?.business_type)}</p>
                            </li>
                        ))}
                    </ul>
                )}
                <LightgalleryProvider
                    lightgallerySettings={{
                        download: false,
                        thumbnail: false,
                        fullScreen: false,
                        share: false
                    }}
                >
                    <Row>
                        {filteredData && filteredData?.length > 0 &&
                            filteredData.map((item, index) => (
                                <Col key={index} sm={3}>
                                    <div className="gallery-single">
                                        <PhotoItem image={item.full_path} group="group1" index={index} />
                                    </div>
                                </Col>
                            ))}
                    </Row>
                </LightgalleryProvider>
            </Container>
        </StyledGallery>
    );
};

const StyledSelect = styled(Select)`
    margin-bottom: 40px;

    .react-select__control {
        border: 1px solid #001a93;
        border-radius: 30px;
        padding: 5px 15px;
        cursor: pointer;
    }

    .react-select__single-value {
        color: #001a93;
        font-size: 16px;
        font-weight: 600;
    }

    .react-select__indicator-separator{
        display: none !important;
    }
    .react-select__option {
        color: #001a93;
        font-size: 16px;
        font-weight: 600;
        cursor: pointer;

        &--is-selected {
            background-color: #001a93;
            color: white;
        }

        &--is-focused {
            background-color: rgba(0, 26, 147, 0.8);
        }
    }

`;

const StyledGallery = styled.section`
    position: relative;
    background-color: #f3f3f3;

    ul {
        display: flex;
        flex-wrap: wrap;
        gap: 15px;
        margin-bottom: 40px;
        li {
            display: flex;
            flex: 1 1 calc(20% - 20px);
            align-items: center;
            justify-content: center;
            padding-top: 15px;
            padding-bottom: 15px;
            border: 1px solid #001a93;
            cursor: pointer;
            position: relative;
            overflow: hidden;
            max-width: 20%;

            p {
                color: #001a93;
                text-align: center;
                font-size: 16px;
                font-style: normal;
                font-weight: 600;
                line-height: 24px;
                position: relative;
                z-index: 22;
            }
            &:after {
                content: '';
                position: absolute;
                bottom: 0;
                left: 0;
                width: 100%;
                height: 0%;
                background-color: #001a93;
                transition: height 0.2s ease-in-out;
                z-index: 1;
            }

            &.active {
                p {
                    color: #ffffff !important;
                }

                &:after {
                    height: 100%;
                    transition: height 0.2s ease-in-out;
                }
            }

            &:hover {
                p {
                    color: #ffffff !important;
                }

                &:after {
                    height: 100%;
                    transition: height 0.2s ease-in-out;
                }
            }
        }
    }

    .col-sm-3 {
        padding: 0 8px;
        margin-bottom: 17px;
    }

    .gallery-single {
        padding-top: calc(300 / 280 * 100%);
        position: relative;
        cursor: pointer;
        overflow: hidden;

        .image-wrapper {
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            transition: transform 0.3s ease-in-out;
        }

        &:hover {
            .image-wrapper {
                transform: scale(1.1);
            }
        }

        img {
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            object-fit: cover;
        }
    }

    @media (max-width: 767px) {
        .col-sm-3 {
            flex: 0 0 50%;
            max-width: 50%;
            padding: 0 5px;
            margin-bottom: 10px;
        }
    }
`;

export default Gallery;